/* osgCompute - Copyright (C) 2008-2009 SVT Group
*
* This library is free software; you can redistribute it and/or modify
* it under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 3 of
* the License, or (at your option) any later version.
*
* This library is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU Lesse General Public License for more details.
*
* The full license is in LICENSE file included with this distribution.
*/

/**************************************************************
* ABOUT THIS DEMO-APPLICATION: GL Post Processing
* ============================
* This application demonstrates the use of osgCompute/osgCuda
* within the scene graph. First, the scene is rendered to a
* texture (see createPreRenderSubGraph) and then post-processed
* by a computation (which applies either a gauss or a sobel
* filter). The resulting image is then transferred back to
* the GL context and is displayed by a screen aligned quad.
***************************************************************/

#include <osg/ArgumentParser>
#include <osg/Texture2D>
#include <osg/Vec4ub>
#include <osg/Geometry>
#include <osg/MatrixTransform>
#include <osgDB/ReadFile>
#include <osgDB/FileUtils>
#include <osgDB/Registry>
#include <osgViewer/Viewer>
#include <osgViewer/ViewerEventHandlers>

#include <osgCuda/Program>
#include <osgCuda/Texture>

#include "TexStreamer"


osg::MatrixTransform* _loadedModelTransform = NULL;
osg::Node* _loadedModel = NULL;

osg::Group* createPreRenderSubGraph(osg::Texture2D* renderTexture,                                                                   
                                   unsigned int samples, unsigned int colorSamples)
{   
   // create a group to contain the flag and the pre rendering camera.
   osg::Group* parent = new osg::Group;

   // create a transform to spin the model.
   _loadedModelTransform = new osg::MatrixTransform;
   _loadedModelTransform->addChild(_loadedModel);

   int tex_width = renderTexture->getTextureWidth();
   int tex_height = renderTexture->getTextureHeight();   

   // then create the camera node to do the render to texture
   {    
      osg::Camera* camera = new osg::Camera;

      // set up the background color and clear mask.
      camera->setClearColor(osg::Vec4(0,0,0,1.0f));
      camera->setClearMask(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

      const osg::BoundingSphere& bs = _loadedModelTransform->getBound();     

      float znear = 1.0f*bs.radius();
      float zfar  = 3.0f*bs.radius();

      // 2:1 aspect ratio as per flag geometry below.
      float proj_top   = 0.25f*znear;
      float proj_right = 0.5f*znear;

      znear *= 0.9f;
      zfar *= 1.1f;

      // set up projection.
      camera->setProjectionMatrixAsFrustum(-proj_right,proj_right,-proj_top,proj_top,znear,zfar);

      // set view
      camera->setReferenceFrame(osg::Transform::ABSOLUTE_RF);
      camera->setViewMatrixAsLookAt(bs.center()-osg::Vec3(0.0f,2.0f,0.0f)*bs.radius(),bs.center(),osg::Vec3(0.0f,0.0f,1.0f));

      // set viewport
      camera->setViewport(0,0,tex_width,tex_height);

      // set the camera to render before the main camera.
      camera->setRenderOrder(osg::Camera::PRE_RENDER);

      // tell the camera to use OpenGL frame buffer object where supported.
      camera->setRenderTargetImplementation(osg::Camera::FRAME_BUFFER_OBJECT);

      // attach the texture and use it as the color buffer.
      camera->attach(osg::Camera::COLOR_BUFFER, renderTexture, 
         0, 0, false,
         samples, colorSamples);

      // add subgraph to render
      camera->addChild(_loadedModelTransform);
      parent->addChild(camera);
   }    

   // rotate the model automatically 
   osg::NodeCallback* nc = new osg::AnimationPathCallback(
       _loadedModelTransform->getBound().center(),osg::Vec3(0.0f,0.0f,1.0f),osg::inDegrees(45.0f));
   _loadedModelTransform->setUpdateCallback(nc);
   
   return parent;
}


osg::Geode* getTexturedQuad( osg::Texture2D& targetTexture )
{
   osg::Geode* geode = new osg::Geode;
   geode->setName("quad");

   osg::Vec3 llCorner = osg::Vec3(-0.5,0,-0.5);
   osg::Vec3 width = osg::Vec3(1,0,0);
   osg::Vec3 height = osg::Vec3(0,0,1);

   //////////
   // QUAD //
   //////////
   osg::Geometry* geom = osg::createTexturedQuadGeometry( llCorner, width, height );
   geode->addDrawable( geom );
   geode->getOrCreateStateSet()->setTextureAttributeAndModes( 0, &targetTexture, osg::StateAttribute::ON );

   return geode;
}


int main(int argc, char *argv[])
{
   osg::setNotifyLevel( osg::WARN );


   osgViewer::Viewer viewer;
   viewer.setUpViewInWindow( 50, 50, 640, 480);
   viewer.getCamera()->setClearColor( osg::Vec4(0.15, 0.15, 0.15, 1.0) );

   osgCuda::setupOsgCudaAndViewer( viewer );

   // load the nodes from the command line arguments.
   osg::ArgumentParser arguments(&argc,argv);
   _loadedModel = osgDB::readNodeFiles(arguments);

   // if not loaded assume no arguments passed in, try use default model instead.
   if (!_loadedModel) 
       _loadedModel = osgDB::readNodeFile("cow.osg");

   if (!_loadedModel)
   {
       osg::notify(osg::NOTICE) << "main(): Could not open model \"cow.osg\" scene." << std::endl;
       return 1;
   }

   ///////////////
   // RESOURCES //
   ///////////////   
   osg::ref_ptr< osgCuda::Texture2D > sourceTexture = new osgCuda::Texture2D;
   sourceTexture->setTextureSize(512, 512);
   sourceTexture->setInternalFormat(GL_RGBA);
   sourceTexture->setSourceType( GL_UNSIGNED_BYTE );
   sourceTexture->setName( "srcBuffer" );
   sourceTexture->setFilter(osg::Texture2D::MIN_FILTER,osg::Texture2D::LINEAR);
   sourceTexture->setFilter(osg::Texture2D::MAG_FILTER,osg::Texture2D::LINEAR);
   sourceTexture->setUsage( osgCompute::GL_TARGET_COMPUTE_SOURCE );
   sourceTexture->addIdentifier( "SRC_BUFFER" );


   osg::ref_ptr< osgCuda::Texture2D > targetTexture = new osgCuda::Texture2D;
   targetTexture->setTextureSize(512, 512);
   targetTexture->setInternalFormat(GL_RGBA);
   targetTexture->setSourceType( GL_UNSIGNED_BYTE );
   targetTexture->setName( "trgBuffer" );
   targetTexture->setFilter(osg::Texture2D::MIN_FILTER,osg::Texture2D::LINEAR);
   targetTexture->setFilter(osg::Texture2D::MAG_FILTER,osg::Texture2D::NEAREST);  
   targetTexture->addIdentifier( "TRG_BUFFER" );

   //////////////////
   // MODULE SETUP //
   //////////////////
   TexDemo::TexStreamer* texStreamer = new TexDemo::TexStreamer;
   texStreamer->setName( "TexStreamer Example" );
   //texStreamer->setFilter( TexDemo::TexStreamer::GAUSS5x5_FILTER );
   texStreamer->setFilter( TexDemo::TexStreamer::SOBEL_FILTER );

   osgCuda::Program* computation = new osgCuda::Program;
   //computation->setComputeOrder( osgCompute::Computation::POST_COMPUTE_PRE_TRAVERSAL );
   computation->addResource( *(sourceTexture->getMemory()) );
   // not necessary, because targetTexture is located in the computation subgraph 
   // and thus it is automatically added to the resources
   //computation->addResource( *(targetTexture->getOrCreateBuffer()) );
   computation->addComputation( *texStreamer );
   
   osg::Node* preRenderScene = createPreRenderSubGraph(sourceTexture, 0, 0);
   computation->addChild( preRenderScene );
   // targetTexture is located in the subgraph of the computation
   computation->addChild( getTexturedQuad( *targetTexture ) );

   ///////////
   // SCENE //
   ///////////
   // translate the original model - just for illustration purpose
   osg::MatrixTransform* sceneTransform = new osg::MatrixTransform;
   osg::Matrix m;
   m.makeScale( 0.2, 0.2, 0.2 );
   m.setTrans(-2.0, 0.0, 0.0 );
   sceneTransform->setMatrix( m );
   sceneTransform->addChild( _loadedModelTransform );
   
   
   osg::Group* scene = new osg::Group;
   scene->addChild( sceneTransform );
   scene->addChild( computation );
   viewer.setSceneData( scene );

   // collect resources
   osg::ref_ptr<osgCompute::ResourceVisitor> rv = new osgCompute::ResourceVisitor;
   rv->apply( *scene );

   viewer.addEventHandler(new osgViewer::StatsHandler);

   return viewer.run();
}
