/* osgCompute - Copyright (C) 2008-2009 SVT Group
 *                                                                     
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 3 of
 * the License, or (at your option) any later version.
 *                                                                     
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesse General Public License for more details.
 *
 * The full license is in LICENSE file included with this distribution.
*/
#include <osg/Notify>
#include "TexStreamer"
        
extern "C"
void filter( const dim3& numBlocks, const dim3& numThreads, void* trgBuffer, void* srcBuffer, unsigned int srcBufferSize, unsigned int filter );

namespace TexDemo
{
    //////////////////////////////////////////////////////////////////////////////////////////////////
    // PUBLIC FUNCTIONS //////////////////////////////////////////////////////////////////////////////
    //////////////////////////////////////////////////////////////////////////////////////////////////
    //------------------------------------------------------------------------------  
    bool TexStreamer::init() 
    { 
        if( !_trgBuffer || !_srcBuffer )
        {
            osg::notify( osg::WARN ) << "TexDemo::TexStreamer::init(): params are missing."
                                     << std::endl;
            return false;
        }

        /////////////////////////
        // COMPUTE KERNEL SIZE //
        /////////////////////////
        // texture size must be a multiple of 16x16 texels
        _threads = dim3( 16, 16, 1 );
        _blocks = dim3( _trgBuffer->getDimension(0)/16, 
                        _trgBuffer->getDimension(1)/16, 1 );

        // Do not forget to call osgCompute::Computation::init()!!!
        return osgCompute::Computation::init();
    }
  
    //------------------------------------------------------------------------------  
    void TexStreamer::launch()
    {
        if( isClear() )
            return;

        filter(  _blocks, 
                 _threads,
                 _trgBuffer->map(),
                 _srcBuffer->map( osgCompute::MAP_DEVICE_SOURCE ),
                 _srcBuffer->getByteSize(),
                 _filter );

        // Below CUDA 3.2 you have to call unmap because
        // OpenSceneGraph does not notify the texture when bound
        // as a frame buffer object. 
        // This step calls cudaGraphicsUnmapResources() so that
        // it can be used as a render target again.
        _srcBuffer->unmap();
    }


    //------------------------------------------------------------------------------
    void TexStreamer::acceptResource( osgCompute::Resource& resource )
    {
        if( resource.isIdentifiedBy( "TRG_BUFFER" ) )
            _trgBuffer = dynamic_cast<osgCompute::Memory*>( &resource );
        if( resource.isIdentifiedBy( "SRC_BUFFER" ) )
            _srcBuffer = dynamic_cast<osgCompute::Memory*>( &resource );
    }

    //////////////////////////////////////////////////////////////////////////////////////////////////
    // PROTECTED FUNCTIONS ///////////////////////////////////////////////////////////////////////////
    //////////////////////////////////////////////////////////////////////////////////////////////////
    //------------------------------------------------------------------------------  
    void TexStreamer::clearLocal() 
    { 
        _threads = dim3(0,0,0);
        _blocks = dim3(0,0,0);
        _trgBuffer = NULL;
        _srcBuffer = NULL;
        _filter = SOBEL_FILTER;
    }
}
